<?php
/**
 * API base abstract class
 *
 * @package    Sms_Net_Gr
 * @subpackage Sms_Net_Gr/admin
 * @link       https://www.smsnet.gr
 * @author     smsnet.gr <info@smsnet.gr>
 * @since 1.0.0
 */

/**
 * Abstract class that encapsulates the business logic for the singleton
 * class that will extend it.
 *
 * @package    Sms_Net_Gr
 * @subpackage Sms_Net_Gr/includes/api
 * @author     smsnet.gr <info@smsnet.gr>
 * @since 1.0.0
 */
abstract class Sms_Net_Gr_Api_Base {
	/**
	 * The API base url
	 *
	 * @var string
	 * @since 1.0.0
	 */
	const BASE_URL = 'https://smsnet.gr/api';

	/**
	 * Get the API key
	 *
	 * @since		1.0.0
	 * @access	public
	 */
	public function get_key() {
		// Get username,password, $reset flag from $_POST.
		$username = $this->get_if_set( $_POST['username'] );
		$password = $this->get_if_set( $_POST['password'] );
		$reset = $this->get_if_set( $_POST['reset'] );

		// Return false if either is null.
		if ( null === $username || null === $password ) {
			return false;
		}
		// Prepare remote url to get the key.
		$action = 'true' === $reset ? '/key/reset' : '/key/get';
		$url = self::BASE_URL . $action . '?username=' . $username ;
		$url .= '&password=' . $password . '&type=json';
		// Get the response from the API.
		$response = $this->make_api_call( $url );
		// Handle the response.
		$response = $this->response_handler( $response, array( 'status', 'key', 'remarks', 'error' ) );
		if( 200 === $response[ 'status_code' ] ) {
			// Get plugin options from wp_options db table.
			$options = get_option( 'sms_net_gr' );
			// Get the api-key from response.
			$key = $this->get_if_set( $response['body']['key'] );
			// Check if not null and update the api-key in wp_options table
			if( null !== $key ) {
				$options['api-key' ] = $key;
				update_option( 'sms_net_gr', $options, true );
			}
		}
		echo json_encode( $response );
		die();
	}

	/**
	 * Get the current balance
	 *
	 * @since		1.0.0
	 * @access	public
	 */
	public function get_balance() {
		// Get the API key from the plugin options.
		$api_key = get_option( 'sms_net_gr' )['api-key'];
		// Prepare the url for the API call.
		$url = self::BASE_URL . '/balance/get?key=' . $api_key . '&type=json';
		// Get the response from the API.
		$response = $this->make_api_call( $url );
		// Handle the response.
		$response = $this->response_handler( $response, array( 'status', 'balance', 'remarks', 'error' ) );
		echo json_encode( $response );
		die();
	}

	/**
	 * Returns all messages whose status has been changed
	 *
	 * @since		1.0.0
	 * @access	public
	 */
	public function get_status() {
		// Get the API key from the plugin options.
		$api_key = get_option( 'sms_net_gr' )['api-key'];
		// Prepare the url for the API call.
		$url = self::BASE_URL . '/status/get?key=' . $api_key . '&type=json';
		// Get the response from the API.
		$response = $this->make_api_call( $url );
		// Handle the response.
		$response = $this->response_handler( $response, array( 'status' ), true );
		echo json_encode( $response );
		die();
	}

	/**
	 * Function to send SMS messages
	 *
	 * @since		1.0.0
	 * @access	public
	 * @param  string $from     sender id
	 * @param  string $to       recipient mobile phone number
	 * @param  string $message  the sms message content
	 * @param  string $encoding the sms message encoding (GSM/UCS)
	 * @param  string $order_id the woocommerce order id (if applicable)
	 */
	public function send_sms( $from = null, $to = null, $message = null, $encoding = null, $order_id = null ) {
		$ajax = null;
		//Check $_POST for parameters
		if( isset( $_POST['from'] ) ) $from = $_POST['from'];
		if( isset( $_POST['to'] ) ) $to = $_POST['to'];
		if( isset( $_POST['message'] ) ) $message = $_POST['message'];
		if( isset( $_POST['encoding'] ) ) $encoding = $_POST['encoding'];
		if( isset( $_POST['ajax'] ) ) $ajax = $_POST['ajax'];
		//Get the API key from the plugin options.
		$api_key = get_option( 'sms_net_gr' )['api-key'];
		// Prepare the url for the API call.
		$url = self::BASE_URL . '/sms/send?key=' . $api_key;
		$url .= '&to=' . $to . '&from=' . urlencode($from) . '&text=' . urlencode($message).
						($encoding == '0' ? '&ucs=true' : '') .
						//'&callback=' . urlencode(get_site_url() . '/') . '&type=json';
						'&callback=' .urlencode('http://139.91.207.17') . '&type=json';

		// Get the response from the API.
		$response = $this->make_api_call( $url );
		// Handle the response
		$response = $this->response_handler( $response, array( 'status', 'id', 'balance', 'remarks' ), true );
		// Prepare database record
		$db_record = array(
			'smsnet_id' => $response['body']->{'id'},
			'send_tstamp' => current_time('mysql', 1),
			'sender_id' => $from,
			'to' => $to,
			'message' => $message,
			'status' => 's',
			'status_tstamp' => current_time('mysql', 1),
			'gsm_encoding' => $encoding,
			'order_id' => $order_id === null ? '-' : $order_id
		);
		// Create database record
		Sms_Net_Gr_Model::instance()->create($db_record);
		if($ajax === 'true') {
			echo json_encode( $response );
			die();
		}
	}

	/**
	 * Function to make the API Calls
	 *
	 * @since	 1.0.0
	 * @access public
	 * @param  string $url the url to make the request
	 * @return array       custom response array
	 */
	public function make_api_call( $url ) {
		$body = '';
		// GET request for the key.
		$response = wp_remote_get( $url );
		// Log if WP_Error
		if( is_wp_error( $response ) ) {
			$status_code = 500;
		} else {
			// Get HTTP status code
			$status_code = wp_remote_retrieve_response_code( $response );
			$body = wp_remote_retrieve_body( $response );
		}
		// Return custom response
		return array(
			'status_code' => $status_code,
			'body' => $body,
		);
	}

	/**
	 * Function to parse the response.
	 * It returns an array of body elements
	 * specified in the $elements parameter.
	 *
	 * @since	 1.0.0
	 * @access public
	 * @param  array $response   the API call response.
	 * @param  array $elements   elements to extract from the response.
	 * @param  bool  $status     whether it is a status API call response.
	 * @return array $response	 processed response array.
	 */
	public function response_handler( $response, $elements = array( 'status' ), $status = false ) {
		if ( 200 == $response[ 'status_code' ] ) {
			$json = json_decode( $response[ 'body' ] );
			// If response contains status just return the JSON.
			if ( $status ) {
				$response[ 'body' ]  = $json;
			} else {
				// If not an error in API response.
				if ( '0' === $json->{'error'} ) {
					$body = array();
					foreach ( $elements as $key ) {
						$body[ $key ] = $json->{$key};
					}
					// Set body
					$response[ 'body' ] = $body;
				} else {
					// Just set status code 401.
					$response[ 'status_code' ]  = 401;
				}
			}
		}
		Sms_Net_Gr_Logger::instance()->log($response);
		return $response;
	}

	/**
	 * Rewrite tags in order to get params from callback.
	 *
	 * @since  1.0.0
	 */
	public function rewrite_tags() {
		add_rewrite_tag( '%msgid%', '([^&]+)' );
		add_rewrite_tag( '%status%', '([^&]+)' );
	}

	/**
	 * Parse callback response in order to
	 * update the sms status.
	 *
	 * @since  1.0.0
	 */
	public function parse_callback() {
		global $wp_query;
		// Get msgid from url
    $msgid = $wp_query->get( 'msgid' );
		// Get status from url
		$status = $wp_query->get( 'status' );

		// Return if either is false
		if ( ! $msgid ||  ! $status) {
        return;
    }

		// Prepara array of data to update
		$data = array('status'=> $status,
									'status_tstamp' => current_time('mysql', 1) );
		// Prepare array of where conditions
		$where = array( 'smsnet_id' => $msgid);

		// Update the record in database
		Sms_Net_Gr_Model::instance()->update($data, $where);
	}

	/**
	 * Helper function to check whether a variable is set.
	 * Probably sjould be moved to another helper class
	 *
	 * @since  1.0.0
	 * @access protected
	 * @param  mixed $value   value to check.
	 * @param  null  $default the default value is null.
	 */
	public function get_if_set( &$value, $default = null ) {
		return isset( $value ) ? $value : $default;
	}

}
