<?php
/**
 * An abstract class definition that is intended to provide basic CRUD
 * and database functionality to classes that iherit from it.
 *
 * @package    Sms_Net_Gr
 * @subpackage Sms_Net_Gr/includes/model
 * @link       https://www.smsnet.gr
 * @author     smsnet.gr <info@smsnet.gr>
 * @since 1.0.0
 */

/**
 * This is used to get/set database table name and to get,create,update,
 * delete table records.
 *
 * @since      1.0.0
 * @package    Sms_Net_Gr_Base_Model
 * @subpackage Sms_Net_Gr/includes/model
 * @author     smsnet.gr <info@smsnet.gr>
 */
abstract class Sms_Net_Gr_Base_Model {
	/**
	 * Table name will be set by inheriting class
	 *
	 * @since 1.0.0
	 * @var string
	 */
	static $table_name = '';

	/**
	 * Function get_table_name
	 *
	 * Returns the table name
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string
	 */
	public static function get_table_name() {
		global $wpdb;
		return $wpdb->prefix . static::$table_name;
	}

	/**
	 * Function set_table_name
	 *
	 * Returns the table name
	 *
	 * @since 1.0.0
	 * @access public
	 * @param  string $name table name.
	 * @return bool
	 */
	public static function set_table_name( $name ) {
		if ( '' !== $name ) {
			 static::$table_name = $name;
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Function all
	 *
	 * Returns entire result set from table
	 *
	 * @since 1.0.0
	 * @access public
	 * @param  string $conditions	default=''.
	 * @return bool
	 */
	public static function all( $conditions = '' ) {
		global $wpdb;
		$all_data = wp_cache_get( self::get_table_name() . '-all' );
		if ( false === $all_data ) {
			$all_data = $wpdb->get_results( 'SELECT * FROM ' . self::get_table_name() . $conditions, ARRAY_A ); // db call ok.
			wp_cache_set( self::get_table_name() . '-all', $all_data );
		}
		self::log_if_error();
		return $all_data;
	}

	/**
	 * Function row
	 *
	 * Returns table row(s) based on conditions
	 *
	 * @since 1.0.0
	 * @access public
	 * @param  string $conditions	default=''.
	 * @return mixed
	 */
	public function row( $conditions = '' ) {
			global $wpdb;
			$rows = wp_cache_get( self::get_table_name() . '-rows' );
			if ( false === $rows ) {
				 $rows = $wpdb->get_results( 'SELECT * FROM `self::get_table_name()` `$conditions`', 'ARRAY_A' ); // db call ok.
				 wp_cache_set( self::get_table_name() . '-rows', $rows );
			}
			self::log_if_error();
			return $rows;
	}

	/**
	 * Function find_by_id
	 *
	 * Returns a single record by id
	 *
	 * @since 1.0.0
	 * @access public
	 * @param  mixed $id id of the record.
	 * @return mixed
	 */
	public function find_by_id( $id ) {
		global $wpdb;
		$record = wp_cache_get( self::get_table_name() . '-record-by-id' );
		if ( false === $rows ) {
			$record = $wpdb->get_results( $wpdb->prepare( 'SELECT * FROM `self::get_table_name()` WHERE id = %d', $id ) ); // db call ok.
			wp_cache_set( self::get_table_name() . '-record-by-id', $rows );
		}
		self::log_if_error();
		return $record;
	}

	/**
	 * Function find
	 *
	 * Returns row(s) based on type and conditions
	 *
	 * @since 1.0.0
	 * @access public
	 * @param  string $type				all|row.
	 * @param  array  $parameters array of AND parameters.
	 * @return mixed
	 */
	public static function find( $type, $parameters ) {
		if ( empty( $type ) ) {
			$type = 'all';
		}
		$conditions = isset( $parameters['conditions'] ) ? (' WHERE ' . join( ' AND ',$parameters['conditions'] )) : null;
		if ( 'all' === $type ) {
			$result = self::all( $conditions );
		} elseif ( 'row' === $type ) {
			$result = self::row( $conditions );
		}
		self::log_if_error();
		return $result;
	}

	/**
	 * Function create
	 *
	 * Inserts data into table
	 *
	 * @since 1.0.0
	 * @access public
	 * @param  array $data	data to insert (in column => value pairs).
	 * @return mixed
	 */
	public static function create( $data ) {
		global $wpdb;
		$wpdb->insert( self::get_table_name(), $data ); // db call ok.
		self::log_if_error();
	}

	/**
	 * Function update
	 *
	 * Updates a row in the table
	 *
	 * @since 1.0.0
	 * @access public
	 * @param  array        $data 				data to update (in column => value pairs).
	 * @param  array        $where 				where conditions (in column => value pairs).
	 * @param  array|string $format 			default=null. An array of formats to be mapped to each
	 *                                 		of the values in $data: %d', '%f', '%s' (integer, float, string).
	 * @param  array|string $where_format	default=null. An array of formats to be mapped to each
	 *                                    of the values in $where: %d', '%f', '%s' (integer, float, string).
	 * @return int|false
	 */
	public static function update( $data, $where, $format = null, $where_format = null ) {
		global $wpdb;
		$wpdb->update( self::get_table_name(), $data, $where, $format, $where_format ); // db call ok; no-cache ok.
		self::log_if_error();
	}

	/**
	 * Function delete
	 *
	 * Deletes a row in the table
	 *
	 * @since 1.0.0
	 * @access public
	 * @param  array        $where	where conditions (in column => value pairs).
	 * @param  array|string $format default=null. An array of formats to be mapped to each
	 *                              of the values in $where: %d', '%f', '%s' (integer, float, string).
	 * @return int|false
	 */
	public static function delete( $where, $format = null ) {
		global $wpdb;
		$wpdb->delete( self::get_table_name(), $where, $format ); // db call ok; no-cache ok.
		self::log_if_error();
	}

	/**
	 * Function record_count
	 *
	 * Returns a count of records
	 *
	 * @since 1.0.0
	 * @access public
	 * @return null|string
	 */
	public static function record_count($conditions = null) {
		global $wpdb;
		$result = $wpdb->get_var( 'SELECT COUNT(*) FROM ' . self::get_table_name() . $conditions ); // db call ok; no-cache ok.
		self::log_if_error();
		return $result;
	}

	/**
	 * Function to log last database error in debug.log
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected static function log_if_error() {
		global $wpdb;
		if ( $wpdb->last_error ){
			Sms_Net_Gr_Logger::instance()->log( $wpdb->last_error );
		}
	}
}
