<?php
/**
 * Provides a list of sent SMS by inplementing 
 * the WP_List_Table class
 *
 * @package    Sms_Net_Gr
 * @subpackage Sms_Net_Gr/includes/tools
 * @link       https://www.smsnet.gr
 * @author     smsnet.gr <info@smsnet.gr>
 * @since 1.0.0
 */

/**
 * Shows sent SMS history in a table and provides sorting, searching and
 * pagination of table records
 *
 * @package    Sms_Net_Gr
 * @subpackage Sms_Net_Gr/includes/tools
 * @link       https://www.smsnet.gr
 * @author     smsnet.gr <info@smsnet.gr>
 * @since 1.0.0
 */

/**
 * Check if WP_List_Table eists
 * otherwise load it.
 */
if ( ! class_exists( 'WP_List_Table' ) ) {
	 require_once( ABSPATH . 'wp-admin/includes/class-wp-list-table.php' );
}

class Sms_Net_Gr_List extends WP_List_Table {

	/**
   * SMS statuses array
   *
   * @var array
   * @since 1.0.0
   */
	public $sms_statuses = null;


	/**
	 * The class constructor
	 *
	 * @since 	1.0.0
	 * @access	public
	 */
	public function __construct() {
		/**
	 * Set values in parent constructor
		 *
	 * @since 1.0.0
	 */
		parent::__construct(
			[
				'singular' => __( 'Message', 'smspc' ), // singular name of the listed records.
				'plural'   => __( 'Messages', 'smspc' ), // plural name of the listed records.
				'ajax'     => false, // should this table support ajax?
			]
		);

		$this->sms_statuses = array(	's' => __( 'Sent', 'sms_net_gr' ),
																	'd' => __('Delivered', 'sms_net_gr'),
																	'f' => __('Failed', 'sms_net_gr'),
																	'b' => __('Insufficient Balance', 'sms_net_gr') );
	}

	/**
	 * Displays custom message when no items are found_posts
	 *
	 * @since 	1.0.0
	 * @access	public
	 */
	public function no_items() {
		_e( 'No messages found.', 'smspc' );
	}

	/**
	 * Retrieve SMS from the database
	 *
	 * @since 	1.0.0
	 * @access	public
	 * @param 	int $per_page
	 * @param 	int $page_number
	 * @return 	array
	 */
	public function get_messages( $per_page = 5, $page_number = 1) {
		global $wpdb;
		$sql = '';
		$search = null;
		if(isset($_REQUEST['s'])) {
			$search = esc_sql($_REQUEST['s']);
		}

		if( null != $search ) {
			$search = esc_sql( $search );
			$status = array_search(strtolower($search), array_map('strtolower', $this->sms_statuses));

			if (DateTime::createFromFormat('Y-m-d', $search) !== FALSE) {
				$sql = " WHERE `send_tstamp` BETWEEN '" . $search .
								" 00:00:00' AND '" . $search ." 23:59:59'";
			} else {
				$sql = " WHERE `smsnet_id` LIKE '%" . $search . "%' OR ";
				$sql .= "`sender_id` LIKE '%" . $search . "%' OR ";
				$sql .= "`to` LIKE '%" . $search . "%' OR ";
				$sql .= "`message` LIKE '%" . $search . "%' OR ";
				if( $status ) {
					$sql .= "`status` LIKE '%" . $status . "%' OR ";
				}
				$sql .= "`order_id` LIKE '%" . $search . "%'";
			}
		}
		if ( isset( $_REQUEST['orderby'] ) ) {
			$order_by = esc_sql( $_REQUEST['orderby'] );
			if( $order_by === 'to') $order_by = '`to`';
			$sql .= ' ORDER BY ' . $order_by . ' ' . (isset($_REQUEST['order']) ? esc_sql($_REQUEST['order']) : 'DESC');
		} else {
			$sql .= ' ORDER BY `send_tstamp` DESC';
		}
		$total_items = Sms_Net_Gr_Model::instance()->record_count( $sql );
		$sql .= " LIMIT $per_page";
		$sql .= ' OFFSET ' . ( $page_number - 1 ) * $per_page;
		$result = Sms_Net_Gr_Model::instance()->all( $sql );

		return [ $result, $total_items ];
	}

	/**
	 * Handles data query and filter, sorting, and pagination.
	 *
	 * @since 	1.0.0
	 * @access	public
	 */
	public function prepare_items($search = NULL) {
		$total_items = 0;
		$messages = null;
		$this->_column_headers = $this->get_column_info();
		$per_page     = $this->get_items_per_page( 'messages_per_page', 5 );
		$current_page = $this->get_pagenum();
		$columns = $this->get_columns();
		$hidden = array();
		$sortable = $this->get_sortable_columns();
		$this->_column_headers = array( $columns, $hidden, $sortable );
		$messages = $this->get_messages( $per_page, $current_page );

		$this->items = $this->set_status_names($messages[0]);
		$total_items = $messages[1];

		$this->set_pagination_args(
			[
				'total_items' => $total_items, // WE have to calculate the total number of items
				'per_page'    => $per_page, // WE have to determine how many items to show on a page
			]
		);
	}

	/**
	 * Set SMS status in the table to
	 * human readable values
	 *
	 * @param array $messages array of sms messages fetched from the database
	 * @return array
	 */
	private function set_status_names($messages) {
		foreach ( $messages as &$message ) {
			$message[ 'status' ] = __( $this->sms_statuses[$message[ 'status' ]], 'sms_net_gr');
		}
		return $messages;
	}

	/**
	 * Method for name column
	 *
	 * @since 	1.0.0
	 * @access	private
	 * @param 	array $item an array of DB data
	 * @return 	string
	 */
	private function column_name( $item ) {
		$delete_nonce = wp_create_nonce( 'smspc_delete_message' );
		$title = '<strong>' . $item['smsnet_id'] . '</strong>';
		$actions = [
			'delete' => sprintf( '<a href="?page=%s&action=%s&message=%s&_wpnonce=%s">Delete</a>', esc_attr( $_REQUEST['page'] ), 'delete', absint( $item['ID'] ), $delete_nonce ),
		];
		return $title . $this->row_actions( $actions );
	}

	/**
	 * Render a column when no column specific method exist.
	 *
	 * @since 	1.0.0
	 * @access	public
	 * @param 	array  $item
	 * @param 	string $column_name
	 * @return 	mixed
	 */
	public function column_default( $item, $column_name ) {
		switch ( $column_name ) {
			case 'smsnet_id':
			case 'send_tstamp':
			case 'sender_id':
			case 'to':
			case 'message':
			case 'status':
			case 'order_id':
				return $item[ $column_name ];
			default:
				return print_r( $item, true );
		}
	}

	/**
	 *  Associative array of columns
	 *
	 * @since 	1.0.0
	 * @access	public
	 * @return 	array
	 */
	public function get_columns() {
		$columns = [
			// 'cb'            => '<input type="checkbox" />',
			'smsnet_id'  => __( 'ID', 'smspc' ),
			'send_tstamp'   => __( 'Datetime sent', 'smspc' ),
			'sender_id'     => __( 'Sender', 'smspc' ),
			'to'            => __( 'Recipient', 'smspc' ),
			'message'       => __( 'Message', 'smspc' ),
			'status'        => __( 'Status', 'smspc' ),
			'order_id'      => __( 'Order ID', 'smspc' ),
		];
		return $columns;
	}

	/**
	 * Columns to make sortable.
	 *
	 * @since 	1.0.0
	 * @access	public
	 * @return 	array
	 */
	public function get_sortable_columns() {
		$sortable_columns = array(
			'smsnet_id' => array( 'smsnet_id', true ),
			'send_tstamp' => array( 'send_tstamp', true ),
			'sender_id' => array( 'sender_id', false ),
			'to' => array( 'to', false ),
			'message' => array( 'to', false ),
			'status' => array( 'status', true ),
			'order_id' => array( 'order_id', false ),
		);
		return $sortable_columns;
	}

}
